<?php
// Autocarga de Composer (requiere vlucas/phpdotenv)
require __DIR__ . '/vendor/autoload.php';

use Dotenv\Dotenv;

// Carga variables de entorno
$dotenv = Dotenv::createImmutable(__DIR__);
$dotenv->load();

// Obtener listas de usuarios y contraseñas desde .env
$userList = array_map('trim', explode(',', $_ENV['PANEL_USERS'] ?? ''));
$passList = array_map('trim', explode(',', $_ENV['PANEL_PASSWORDS'] ?? ''));

// Mapear usuarios a sus contraseñas
$CREDENTIALS = array_combine($userList, $passList) ?: [];

// Iniciar sesión
session_start();

// NUEVO: Variable para almacenar errores del editor JSON
$json_editor_error = '';
if (isset($_SESSION['json_editor_error'])) {
    $json_editor_error = $_SESSION['json_editor_error'];
    unset($_SESSION['json_editor_error']);
}

// Procesar formulario de login
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['login'])) {
    $u = $_POST['user'] ?? '';
    $p = $_POST['pass'] ?? '';

    if (isset($CREDENTIALS[$u]) && hash_equals($CREDENTIALS[$u], $p)) {
        $_SESSION['logged_in'] = true;
        header('Location: ' . $_SERVER['PHP_SELF']);
        exit;
    } else {
        $error = 'Usuario o contraseña incorrectos.';
    }
}

// Si no está autenticado, mostramos login
if (empty($_SESSION['logged_in'])):
?>
<!DOCTYPE html>
<html lang="es">
<head>
  <meta charset="UTF-8">
  <title>Login - Panel Tv</title>
  <style>
    body{display:flex;align-items:center;justify-content:center;height:100vh;margin:0;background:#f2f2f2;}
    .box{background:#fff;padding:20px;border-radius:6px;box-shadow:0 0 10px rgba(0,0,0,0.1);}
    input{width:100%;padding:10px;margin:8px 0;border:1px solid #ccc;border-radius:4px;}
    button{width:100%;padding:10px;background:#007bff;color:#fff;border:none;border-radius:4px;cursor:pointer;}
    .error{color:#d9534f;margin-bottom:10px;}
  </style>
</head>
<body>
  <div class="box">
    <h2>Acceso Panel Tv</h2>
    <?php if(!empty($error)): ?><div class="error"><?= htmlspecialchars($error) ?></div><?php endif; ?>
    <form method="post">
      <input type="text" name="user" placeholder="Usuario" required autofocus>
      <input type="password" name="pass" placeholder="Contraseña" required>
      <button type="submit" name="login">Ingresar</button>
    </form>
  </div>
</body>
</html>
<?php
  exit;
endif;


/***************************************
 * Configuraciones iniciales y Encriptación
 **************************************/
// Ruta a los archivos JSON
define('JSON_FILE_PATH', __DIR__ . '/tv.json');
define('JSON_UNENCRYPTED_FILE_PATH', __DIR__ . '/tv2.json'); // Nuevo: Ruta a tv2.json

// Clave y método de encriptación AES
define('ENCRYPTION_KEY', 'e72of82ke0gu2o2k');
define('ENCRYPTION_METHOD', 'aes-128-ecb');

/****************************************************
 * FUNCIONES DE ENCRIPTACIÓN / DESENCRIPTACIÓN
 ****************************************************/

/**
 * Encripta un string usando AES-128-ECB.
 * @param string $data El string a encriptar.
 * @return string String encriptado y codificado en base64.
 */
function encrypt_data($data) {
    if (empty($data)) return '';
    // El IV es opcional para ECB, se puede pasar como '' o null.
    $encrypted = openssl_encrypt($data, ENCRYPTION_METHOD, ENCRYPTION_KEY, 0, '');
    return $encrypted;
}

/**
 * Desencripta un string. Es capaz de manejar datos doblemente encriptados
 * por un error en versiones anteriores del script.
 * @param string $data El string encriptado.
 * @return string El string original desencriptado.
 */
function decrypt_data($data) {
    if (empty($data)) {
        return '';
    }

    $current_data = $data;

    // Bucle para intentar desencriptar hasta 2 veces. Esto corrige la doble encriptación.
    for ($i = 0; $i < 2; $i++) {
        // openssl_decrypt espera datos en base64. Si falla, es probable que ya sea texto plano.
        $decrypted_once = openssl_decrypt($current_data, ENCRYPTION_METHOD, ENCRYPTION_KEY, 0, '');
        
        if ($decrypted_once === false) {
            // La desencriptación falló. Asumimos que $current_data es el texto plano final.
            return $current_data;
        }
        
        // La desencriptación fue exitosa, actualizamos el valor y probamos una vez más por si acaso.
        $current_data = $decrypted_once;
    }
    
    return $current_data;
}

// NUEVO: Funciones recursivas para encriptar/desencriptar el array completo para el editor
/**
 * Recorre un array de datos de TV y desencripta los campos necesarios.
 * @param array &$data El array a desencriptar (pasado por referencia).
 */
function decrypt_json_data_recursively(&$data) {
    if (!is_array($data)) return;
    foreach ($data as &$section) {
        if (isset($section['samples']) && is_array($section['samples'])) {
            foreach ($section['samples'] as &$sample) {
                if (isset($sample['url'])) {
                    $sample['url'] = decrypt_data($sample['url']);
                }
                if (isset($sample['drm_license_uri'])) {
                    $sample['drm_license_uri'] = decrypt_data($sample['drm_license_uri']);
                }
                if (isset($sample['headers']) && is_array($sample['headers'])) {
                    foreach ($sample['headers'] as $key => &$value) {
                        $value = decrypt_data($value);
                    }
                    unset($value);
                }
            }
            unset($sample);
        }
    }
    unset($section);
}

/**
 * Recorre un array de datos de TV y encripta los campos necesarios.
 * @param array &$data El array a encriptar (pasado por referencia).
 */
function encrypt_json_data_recursively(&$data) {
    if (!is_array($data)) return;
    foreach ($data as &$section) {
        if (isset($section['samples']) && is_array($section['samples'])) {
            foreach ($section['samples'] as &$sample) {
                if (isset($sample['url'])) {
                    $sample['url'] = encrypt_data($sample['url']);
                }
                if (isset($sample['drm_license_uri'])) {
                    $sample['drm_license_uri'] = encrypt_data($sample['drm_license_uri']);
                }
                if (isset($sample['headers']) && is_array($sample['headers'])) {
                    foreach ($sample['headers'] as $key => &$value) {
                        $value = encrypt_data($value);
                    }
                    unset($value);
                }
            }
            unset($sample);
        }
    }
    unset($section);
}


/****************************************************
 * FUNCIONES DE MANEJO DE ARCHIVO JSON LOCAL
 ****************************************************/

/**
 * Obtiene el contenido de un archivo JSON.
 * Devuelve un array PHP.
 */
function getJsonFromFile($filePath) {
    if (!file_exists($filePath)) {
        // Si es el archivo principal (tv.json), lo creamos. Si no, devolvemos array vacío.
        if ($filePath === JSON_FILE_PATH) {
            file_put_contents($filePath, '[]');
        }
        return [];
    }
    $jsonContent = file_get_contents($filePath);
    $data = json_decode($jsonContent, true);
    return is_array($data) ? $data : [];
}

/**
 * Actualiza un archivo JSON con el nuevo contenido.
 *
 * @param string $filePath    Ruta al archivo JSON.
 * @param array  $dataArray   El array de datos a guardar.
 * @return void
 */
function updateJsonInFile($filePath, $dataArray) {
    $updatedJson = json_encode($dataArray, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
    if (file_put_contents($filePath, $updatedJson) === false) {
        die("Error: No se pudo escribir en el archivo '{$filePath}'. Verifica los permisos.");
    }
}

/****************************************************
 * NUEVAS FUNCIONES PARA DETECTAR JSON SIN ENCRIPTAR
 ****************************************************/

/**
 * Verifica si un string es probablemente Base64 (característica de la encriptación).
 * @param mixed $data El string a verificar.
 * @return bool
 */
function is_base64_encoded($data) {
    if (!is_string($data) || empty(trim($data))) {
        return false;
    }
    // La encriptación produce Base64. Si la decodificación falla, no está encriptado.
    return (base64_decode($data, true) !== false);
}

/**
 * Revisa el archivo tv2.json en busca de campos que no estén encriptados.
 * @param string $filePath Ruta al archivo tv2.json.
 * @return bool True si se encuentra al menos un campo sin encriptar.
 */
function checkForUnencryptedData($filePath) {
    $data = getJsonFromFile($filePath);
    if (empty($data)) {
        return false;
    }

    foreach ($data as $section) {
        if (empty($section['samples'])) continue;
        foreach ($section['samples'] as $sample) {
            // Revisar URL
            if (isset($sample['url']) && !is_base64_encoded($sample['url'])) {
                return true;
            }
            // Revisar Licencia DRM
            if (isset($sample['drm_license_uri']) && !is_base64_encoded($sample['drm_license_uri'])) {
                return true;
            }
            // Revisar Headers
            if (isset($sample['headers']) && is_array($sample['headers'])) {
                foreach ($sample['headers'] as $value) {
                    if (!is_base64_encoded($value)) {
                        return true; // Encontrado un header sin encriptar
                    }
                }
            }
        }
    }
    return false; // No se encontró nada sin encriptar
}

/****************************************************
 * LÓGICA DE REORDENAMIENTO (CATEGORÍAS Y CANALES)
 ****************************************************/

function moveCategoryPosition(&$data, $indexFrom, $indexTo) {
    $indexFrom = intval($indexFrom);
    $indexTo   = intval($indexTo);
    if ($indexFrom < 0 || $indexFrom >= count($data) || $indexTo < 0 || $indexTo >= count($data)) {
        return;
    }
    $item = array_splice($data, $indexFrom, 1);
    array_splice($data, $indexTo, 0, $item);
}

function moveChannelPosition(&$samples, $oldIndex, $newIndex) {
    if ($oldIndex === $newIndex) return;
    if ($oldIndex < 0 || $oldIndex >= count($samples) || $newIndex < 0 || $newIndex >= count($samples)) {
        return;
    }
    $item = array_splice($samples, $oldIndex, 1);
    array_splice($samples, $newIndex, 0, $item);
}

/****************************************************
 * OBTENER EL JSON INICIAL Y VERIFICAR tv2.json
 ****************************************************/

$data = getJsonFromFile(JSON_FILE_PATH);
$unencryptedDataDetected = checkForUnencryptedData(JSON_UNENCRYPTED_FILE_PATH);


/****************************************************
 * PROCESAR ACCIONES DEL USUARIO (POST)
 ****************************************************/
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';

    switch ($action) {
        // NUEVO: Acción para guardar el contenido del editor JSON
        case 'save_raw_json':
            $rawJsonContent = $_POST['json_content'] ?? '[]';
            $decodedData = json_decode($rawJsonContent, true);

            // 1. Validación de sintaxis
            if (json_last_error() !== JSON_ERROR_NONE) {
                $_SESSION['json_editor_error'] = "Error de sintaxis en el JSON: " . json_last_error_msg() . ". No se pudo guardar.";
                // Redirigimos de vuelta al editor para que el usuario no pierda su texto
                $_SESSION['json_editor_content_buffer'] = $rawJsonContent; 
                header("Location: " . $_SERVER['PHP_SELF'] . "?view=editor");
                exit;
            }

            // 2. Re-encriptar los campos necesarios
            encrypt_json_data_recursively($decodedData);

            // 3. Guardar el archivo
            updateJsonInFile(JSON_FILE_PATH, $decodedData);
            // Salimos del modo editor
            header("Location: " . $_SERVER['PHP_SELF']);
            exit;
            break;

        // NUEVA ACCIÓN para encriptar y cargar desde tv2.json
        case 'encrypt_and_merge':
            $dataToProcess = getJsonFromFile(JSON_UNENCRYPTED_FILE_PATH);
            if (!empty($dataToProcess)) {
                // Iteramos sobre todos los datos para asegurar la encriptación
                foreach ($dataToProcess as &$section) {
                    if (empty($section['samples'])) continue;
                    foreach ($section['samples'] as &$sample) {
                        // Desencriptamos y re-encriptamos para normalizar
                        if (isset($sample['url'])) {
                            $sample['url'] = encrypt_data(decrypt_data($sample['url']));
                        }
                        if (isset($sample['drm_license_uri'])) {
                            $sample['drm_license_uri'] = encrypt_data(decrypt_data($sample['drm_license_uri']));
                        }
                        if (isset($sample['headers']) && is_array($sample['headers'])) {
                            foreach ($sample['headers'] as $key => &$value) {
                                $value = encrypt_data(decrypt_data($value));
                            }
                            unset($value);
                        }
                    }
                    unset($sample);
                }
                unset($section);
                // Sobrescribimos el archivo principal (tv.json) con la data procesada
                updateJsonInFile(JSON_FILE_PATH, $dataToProcess);
            }
            break;

        case 'add_section':
            $newSectionName = trim($_POST['section_name']);
            if ($newSectionName !== '') {
                $data[] = ["name" => $newSectionName, "samples" => []];
            }
            break;

        case 'rename_section':
            $sectionIndex = intval($_POST['section_index']);
            $newSectionName = trim($_POST['new_section_name']);
            if (isset($data[$sectionIndex]) && $newSectionName !== '') {
                $data[$sectionIndex]['name'] = $newSectionName;
            }
            break;

        case 'delete_section':
            $sectionIndex = intval($_POST['section_index']);
            if (isset($data[$sectionIndex])) {
                array_splice($data, $sectionIndex, 1);
            }
            break;

        case 'move_section':
            $sectionIndex = intval($_POST['section_index']);
            $newPosition  = intval($_POST['new_position']) - 1;
            moveCategoryPosition($data, $sectionIndex, $newPosition);
            break;

        case 'add_channel':
            $sectionIndex = intval($_POST['section_index']);
            if (isset($data[$sectionIndex])) {
                $newChannel = [
                    "name"  => trim($_POST['channel_name']),
                    "url"   => encrypt_data(trim($_POST['channel_url'])),
                    "type"  => $_POST['channel_type'],
                    "icono" => trim($_POST['channel_icon'])
                ];
                if ($newChannel['type'] === "CLEARKEY") {
                    $newChannel["drm_license_uri"] = encrypt_data(trim($_POST['channel_drm']));
                }
                if (($_POST['has_headers'] ?? 'no') === 'yes') {
                    $headerKeys = $_POST['channel_header_key'] ?? [];
                    $headerVals = $_POST['channel_header_val'] ?? [];
                    $headersObj = [];
                    for ($i = 0; $i < count($headerKeys); $i++) {
                        $k = trim($headerKeys[$i]); $v = trim($headerVals[$i]);
                        if ($k !== '') $headersObj[$k] = encrypt_data($v);
                    }
                    if (!empty($headersObj)) $newChannel["headers"] = $headersObj;
                }
                $data[$sectionIndex]['samples'][] = $newChannel;
            }
            break;

        case 'delete_channel':
            $sectionIndex = intval($_POST['section_index']);
            $channelIndex = intval($_POST['channel_index']);
            if (isset($data[$sectionIndex]['samples'][$channelIndex])) {
                array_splice($data[$sectionIndex]['samples'], $channelIndex, 1);
            }
            break;

        case 'edit_channel':
            $sectionIndex = intval($_POST['section_index']);
            $channelIndex = intval($_POST['channel_index']);
            if (isset($data[$sectionIndex]['samples'][$channelIndex])) {
                $channel =& $data[$sectionIndex]['samples'][$channelIndex];
                $channel['name']  = trim($_POST['channel_name']);
                $channel['url']   = encrypt_data(trim($_POST['channel_url']));
                $channel['icono'] = trim($_POST['channel_icon']);
                $channel['type']  = $_POST['channel_type'];

                if ($channel['type'] === "CLEARKEY") {
                    $channel['drm_license_uri'] = encrypt_data(trim($_POST['channel_drm']));
                } else {
                    unset($channel['drm_license_uri']);
                }
                if (($_POST['has_headers'] ?? 'no') === 'yes') {
                    $headerKeys = $_POST['channel_header_key'] ?? [];
                    $headerVals = $_POST['channel_header_val'] ?? [];
                    $headersObj = [];
                    for ($i = 0; $i < count($headerKeys); $i++) {
                        $k = trim($headerKeys[$i]); $v = trim($headerVals[$i]);
                        if ($k !== '') $headersObj[$k] = encrypt_data($v);
                    }
                    $channel['headers'] = $headersObj;
                } else {
                    unset($channel['headers']);
                }
                if (isset($_POST['channel_new_position'])) {
                    $channelNewPosition = intval($_POST['channel_new_position']) - 1;
                    moveChannelPosition($data[$sectionIndex]['samples'], $channelIndex, $channelNewPosition);
                }
            }
            break;
    }

    // Para todas las acciones del panel normal, actualizamos tv.json
    if ($action !== 'encrypt_and_merge' && $action !== 'save_raw_json') {
        updateJsonInFile(JSON_FILE_PATH, $data);
    }
    
    header("Location: " . $_SERVER['PHP_SELF']);
    exit;
}

// NUEVO: Preparar contenido para el editor JSON si está activo
$json_for_editor = '';
if (isset($_SESSION['json_editor_content_buffer'])) {
    // Si hubo un error de sintaxis, recuperamos el contenido que el usuario intentó guardar
    $json_for_editor = $_SESSION['json_editor_content_buffer'];
    unset($_SESSION['json_editor_content_buffer']);
} else {
    // Si no, cargamos y desencriptamos el contenido de tv.json
    $data_for_editor = getJsonFromFile(JSON_FILE_PATH);
    decrypt_json_data_recursively($data_for_editor);
    $json_for_editor = json_encode($data_for_editor, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
}
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <title>Panel de Administración - TV JSON</title>
    <style>
        body { background: #f0f2f5; margin: 0; padding: 0; font-family: sans-serif; }
        .container { width: 90%; max-width: 1100px; margin: 20px auto; background: #fff; padding: 20px; border-radius: 8px; }
        h1, h2, h3 { text-align: center; margin: 10px 0; }
        .section-box { border: 1px solid #ccc; padding: 10px; margin-bottom: 20px; border-radius: 5px; }
        .section-header { display: flex; justify-content: space-between; align-items: flex-start; }
        .section-title { display: inline-block; margin: 0; }
        .channels-list { margin-left: 20px; margin-top: 10px; list-style: none; padding-left: 0; }
        .channels-list li { margin-bottom: 8px; padding: 5px; border-left: 3px solid #eee; }
        button { cursor: pointer; margin: 2px; }
        form { margin: 0; padding: 0; }
        .inline-form { display: inline-block; }
        .label-input { margin: 6px 0; }
        .input-text { width: 95%; padding: 4px; margin: 4px 0; }
        .header-pair { display: flex; gap: 5px; margin-bottom: 5px; }
        .header-pair input { width: 45%; }
        .add-header-btn { margin: 8px 0; }
        .small { font-size: 0.9em; color: #666; }
        .btn-primary { background: #28a745; color: #fff; border: none; padding: 5px 10px; border-radius: 3px; }
        .btn-danger { background: #dc3545; color: #fff; border: none; padding: 5px 10px; border-radius: 3px; }
        .btn-secondary { background: #007bff; color: #fff; border: none; padding: 5px 10px; border-radius: 3px; }
        .btn-warning { background: #ffc107; color: #333; border: none; padding: 5px 10px; border-radius: 3px; }
        .add-section-form, .add-channel-form, .edit-channel-form { background: #f7f7f7; margin-top: 15px; padding: 10px; border-radius: 5px; border: 1px solid #e0e0e0; }
        .alert-unencrypted {
            background-color: #fff3cd; border: 1px solid #ffeeba; color: #856404;
            padding: 15px; margin-bottom: 20px; border-radius: 5px; text-align: center;
        }
        /* NUEVO: Estilos para el editor JSON */
        .top-controls { display: flex; justify-content: space-between; align-items: center; margin-bottom: 15px; border-bottom: 1px solid #ddd; padding-bottom: 10px; }
        #jsonEditorContainer { display: none; }
        .json-editor-warning { background-color: #f8d7da; border: 1px solid #f5c6cb; color: #721c24; padding: 15px; margin-bottom: 15px; border-radius: 5px; text-align: center; font-weight: bold; }
        #json_content_editor { width: 100%; height: 60vh; font-family: 'Courier New', Courier, monospace; font-size: 14px; border: 1px solid #ccc; border-radius: 4px; padding: 10px; box-sizing: border-box; }
        #findReplaceBar { display: none; background: #f0f0f0; padding: 8px; margin-bottom: 10px; border-radius: 4px; }
        #findReplaceBar input[type="text"] { margin-right: 5px; }
        #findReplaceBar button { margin: 0 2px; }
        #syntaxError { color: #d9534f; margin-bottom: 10px; font-weight: bold; }
    </style>
</head>
<body>
<div class="container">
    <div id="panelNormal">
        <div class="top-controls">
            <h1>Panel de Administración - TV JSON</h1>
            <!-- NUEVO: Botón para mostrar el editor JSON -->
            <button id="showJsonEditorBtn" class="btn-secondary">Ver JSON Normal</button>
        </div>
        <p class="small" style="text-align:center;">Editando archivo local: <strong>tv.json</strong></p>

        <?php if ($unencryptedDataDetected): ?>
        <div class="alert-unencrypted">
            <strong>¡Alerta!</strong> Se ha detectado el archivo <strong>tv2.json</strong> con contenido sin encriptar.
            <br>Presiona el botón para encriptar todo su contenido y cargarlo en el archivo principal (tv.json).
            <form method="POST" style="margin-top:10px;">
                <input type="hidden" name="action" value="encrypt_and_merge">
                <button type="submit" class="btn-warning">Encriptar y Cargar desde tv2.json</button>
            </form>
        </div>
        <?php endif; ?>

        <div class="add-section-form">
            <h3>Agregar nueva categoría</h3>
            <form method="POST">
                <input type="hidden" name="action" value="add_section">
                <div class="label-input">
                    <label>Nombre de la sección:</label><br>
                    <input type="text" name="section_name" class="input-text" required>
                </div>
                <button type="submit" class="btn-primary">Agregar sección</button>
            </form>
        </div>

        <?php foreach ($data as $sectionIndex => $section): ?>
            <div class="section-box">
                <div class="section-header">
                    <div>
                        <h2 class="section-title" style="display:inline;"><?php echo htmlspecialchars($section['name']); ?></h2>
                        <button type="button" class="btn-secondary" onclick="toggleRenameForm('<?php echo $sectionIndex; ?>')">Cambiar nombre</button>
                        <div id="renameForm_<?php echo $sectionIndex; ?>" style="display:none; margin-top:5px;">
                            <form method="POST" class="inline-form">
                                <input type="hidden" name="action" value="rename_section"><input type="hidden" name="section_index" value="<?php echo $sectionIndex; ?>">
                                <input type="text" name="new_section_name" value="<?php echo htmlspecialchars($section['name']); ?>" class="input-text" required>
                                <button type="submit" class="btn-primary">Actualizar</button>
                            </form>
                        </div>
                    </div>
                    <div>
                        <form method="POST" class="inline-form">
                            <input type="hidden" name="action" value="delete_section"><input type="hidden" name="section_index" value="<?php echo $sectionIndex; ?>">
                            <button type="submit" class="btn-danger" onclick="return confirm('¿Estás seguro de eliminar la sección completa?')">Eliminar</button>
                        </form>
                        <form method="POST" class="inline-form">
                            <input type="hidden" name="action" value="move_section"><input type="hidden" name="section_index" value="<?php echo $sectionIndex; ?>">
                            <label>Posición:</label> <input type="number" name="new_position" value="<?php echo ($sectionIndex+1); ?>" style="width:60px;">
                            <button type="submit" class="btn-warning">Mover</button>
                        </form>
                    </div>
                </div>

                <ul class="channels-list">
                    <?php if (!empty($section['samples'])): ?>
                        <?php foreach ($section['samples'] as $channelIndex => $channel): ?>
                            <li>
                                <strong><?php echo htmlspecialchars($channel['name']); ?></strong>
                                <form method="POST" class="inline-form">
                                    <input type="hidden" name="action" value="delete_channel"><input type="hidden" name="section_index" value="<?php echo $sectionIndex; ?>"><input type="hidden" name="channel_index" value="<?php echo $channelIndex; ?>">
                                    <button type="submit" class="btn-danger" onclick="return confirm('¿Eliminar este canal?')">Eliminar</button>
                                </form>
                                <button type="button" class="btn-secondary" onclick="toggleEditForm('<?php echo $sectionIndex; ?>','<?php echo $channelIndex; ?>')">Editar</button>

                                <div id="editForm_<?php echo $sectionIndex; ?>_<?php echo $channelIndex; ?>" class="edit-channel-form" style="display:none;">
                                    <h4>Editar canal</h4>
                                    <form method="POST">
                                        <input type="hidden" name="action" value="edit_channel"><input type="hidden" name="section_index" value="<?php echo $sectionIndex; ?>"><input type="hidden" name="channel_index" value="<?php echo $channelIndex; ?>">
                                        <div class="label-input"><label>Nombre:</label><br><input type="text" name="channel_name" class="input-text" value="<?php echo htmlspecialchars($channel['name']); ?>" required></div>
                                        <div class="label-input"><label>URL del canal:</label><br><input type="text" name="channel_url" class="input-text" value="<?php echo htmlspecialchars(decrypt_data($channel['url'])); ?>"></div>
                                        <div class="label-input"><label>URL del icono:</label><br><input type="text" name="channel_icon" class="input-text" value="<?php echo htmlspecialchars($channel['icono'] ?? ""); ?>"></div>
                                        <div class="label-input"><label>Tipo:</label><br><select name="channel_type" onchange="toggleDrmFieldEdit('<?php echo $sectionIndex; ?>','<?php echo $channelIndex; ?>', this.value)"><option value="HLS" <?php echo ($channel['type'] === 'HLS' ? 'selected' : ''); ?>>HLS</option><option value="CLEARKEY" <?php echo ($channel['type'] === 'CLEARKEY' ? 'selected' : ''); ?>>DRM (CLEARKEY)</option></select></div>
                                        <?php $displayDrm = ($channel['type'] === 'CLEARKEY') ? 'block' : 'none'; $drmVal = isset($channel['drm_license_uri']) ? decrypt_data($channel['drm_license_uri']) : ''; ?>
                                        <div class="label-input" id="drmFieldEdit_<?php echo $sectionIndex; ?>_<?php echo $channelIndex; ?>" style="display: <?php echo $displayDrm; ?>;"><label>URL de la licencia DRM:</label><br><input type="text" name="channel_drm" class="input-text" value="<?php echo htmlspecialchars($drmVal); ?>"></div>
                                        <div class="label-input"><label>Nueva posición:</label><br><input type="number" name="channel_new_position" value="<?php echo $channelIndex+1; ?>" style="width:80px;"></div>
                                        <?php $headersChecked = isset($channel['headers']) && is_array($channel['headers']); ?>
                                        <div><label>¿Headers?</label> <input type="radio" name="has_headers" value="yes" <?php echo $headersChecked ? 'checked' : ''; ?> onclick="document.getElementById('headersEdit_<?php echo $sectionIndex; ?>_<?php echo $channelIndex; ?>').style.display='block';"> Sí <input type="radio" name="has_headers" value="no" <?php echo !$headersChecked ? 'checked' : ''; ?> onclick="document.getElementById('headersEdit_<?php echo $sectionIndex; ?>_<?php echo $channelIndex; ?>').style.display='none';"> No</div>
                                        <div id="headersEdit_<?php echo $sectionIndex; ?>_<?php echo $channelIndex; ?>" style="display: <?php echo $headersChecked?'block':'none'; ?>;"><p>Headers:</p><div id="headerPairsEdit_<?php echo $sectionIndex; ?>_<?php echo $channelIndex; ?>">
                                            <?php if ($headersChecked) { foreach ($channel['headers'] as $hk => $hv) { echo '<div class="header-pair"><input type="text" name="channel_header_key[]" value="'.htmlspecialchars($hk).'" placeholder="Header..." /><input type="text" name="channel_header_val[]" value="'.htmlspecialchars(decrypt_data($hv)).'" placeholder="Valor..." /></div>'; } } else { echo '<div class="header-pair"><input type="text" name="channel_header_key[]" placeholder="Header..." /><input type="text" name="channel_header_val[]" placeholder="Valor..." /></div>'; } ?>
                                        </div><button type="button" class="add-header-btn btn-secondary" onclick="addHeaderPair('headerPairsEdit_<?php echo $sectionIndex; ?>_<?php echo $channelIndex; ?>')">Agregar otro header</button></div>
                                        <div style="margin-top: 10px;"><button type="submit" class="btn-primary">Actualizar canal</button></div>
                                    </form>
                                </div>
                            </li>
                        <?php endforeach; ?>
                    <?php else: ?>
                        <li><em>No hay canales en esta sección.</em></li>
                    <?php endif; ?>
                </ul>

                <div class="add-channel-form">
                    <h4>Agregar nuevo canal a "<?php echo htmlspecialchars($section['name']); ?>"</h4>
                    <form method="POST">
                        <input type="hidden" name="action" value="add_channel"><input type="hidden" name="section_index" value="<?php echo $sectionIndex; ?>">
                        <div class="label-input"><label>Nombre:</label><br><input type="text" name="channel_name" class="input-text" required></div>
                        <div class="label-input"><label>URL del canal:</label><br><input type="text" name="channel_url" class="input-text"></div>
                        <div class="label-input"><label>URL del icono:</label><br><input type="text" name="channel_icon" class="input-text"></div>
                        <div class="label-input"><label>Tipo:</label><br><select name="channel_type" onchange="toggleDrmField(this, '<?php echo $sectionIndex; ?>')"><option value="HLS">HLS</option><option value="CLEARKEY">DRM (CLEARKEY)</option></select></div>
                        <div class="label-input" id="drmField_<?php echo $sectionIndex; ?>" style="display:none;"><label>URL de la licencia DRM:</label><br><input type="text" name="channel_drm" class="input-text"></div>
                        <div><label>¿Headers?</label> <input type="radio" name="has_headers" value="yes" onclick="document.getElementById('headersContainer_<?php echo $sectionIndex; ?>').style.display='block';"> Sí <input type="radio" name="has_headers" value="no" checked onclick="document.getElementById('headersContainer_<?php echo $sectionIndex; ?>').style.display='none';"> No</div>
                        <div id="headersContainer_<?php echo $sectionIndex; ?>" style="display:none;"><p>Headers:</p><div id="headerPairs_<?php echo $sectionIndex; ?>"><div class="header-pair"><input type="text" name="channel_header_key[]" placeholder="Header..." /><input type="text" name="channel_header_val[]" placeholder="Valor..." /></div></div><button type="button" class="add-header-btn btn-secondary" onclick="addHeaderPair('headerPairs_<?php echo $sectionIndex; ?>')">Agregar otro header</button></div>
                        <div style="margin-top: 10px;"><button type="submit" class="btn-primary">Agregar canal</button></div>
                    </form>
                </div>
            </div>
        <?php endforeach; ?>
    </div>

    <!-- NUEVO: Contenedor del Editor JSON -->
    <div id="jsonEditorContainer">
        <div class="top-controls">
            <h1>Editor de JSON Normal</h1>
            <button id="hideJsonEditorBtn" class="btn-secondary">Volver al Panel Normal</button>
        </div>
        <div class="json-editor-warning">
            ¡CUIDADO! Estás editando el archivo JSON directamente. Un error de sintaxis (como una coma faltante o un corchete mal cerrado) puede hacer que la aplicación deje de funcionar. El sistema no guardará si detecta un error de sintaxis.
        </div>
        
        <?php if (!empty($json_editor_error)): ?>
            <div id="syntaxError"><?= htmlspecialchars($json_editor_error) ?></div>
        <?php endif; ?>

        <div id="findReplaceBar">
            <input type="text" id="findInput" placeholder="Buscar...">
            <input type="text" id="replaceInput" placeholder="Reemplazar con...">
            <button id="replaceBtn" class="btn-warning">Reemplazar</button>
            <button id="replaceAllBtn" class="btn-warning">Reemplazar Todo</button>
            <button id="closeFindBtn" class="btn-danger" style="float:right;">X</button>
        </div>

        <form method="POST">
            <input type="hidden" name="action" value="save_raw_json">
            <textarea name="json_content" id="json_content_editor"><?php echo htmlspecialchars($json_for_editor); ?></textarea>
            <div style="text-align: right; margin-top: 15px;">
                <button type="submit" class="btn-primary" style="padding: 10px 20px;">Guardar Cambios</button>
            </div>
        </form>
    </div>

</div>

<script>
// Funciones del panel original
function toggleEditForm(s, c) { var e = document.getElementById('editForm_'+s+'_'+c); e.style.display = (e.style.display === 'none') ? 'block' : 'none'; }
function toggleRenameForm(s) { var e = document.getElementById('renameForm_'+s); e.style.display = (e.style.display === 'none') ? 'block' : 'none'; }
function toggleDrmField(el, s) { document.getElementById('drmField_'+s).style.display = (el.value === 'CLEARKEY') ? 'block' : 'none'; }
function toggleDrmFieldEdit(s, c, v) { document.getElementById('drmFieldEdit_'+s+'_'+c).style.display = (v === 'CLEARKEY') ? 'block' : 'none'; }
function addHeaderPair(id) {
    var c = document.getElementById(id); if (!c) return;
    var d = document.createElement('div'); d.className = 'header-pair';
    d.innerHTML = '<input type="text" name="channel_header_key[]" placeholder="Header..." /> <input type="text" name="channel_header_val[]" placeholder="Valor..." />';
    c.appendChild(d);
}

// NUEVO: Lógica para el editor JSON
document.addEventListener('DOMContentLoaded', function() {
    const panelNormal = document.getElementById('panelNormal');
    const jsonEditorContainer = document.getElementById('jsonEditorContainer');
    const showJsonEditorBtn = document.getElementById('showJsonEditorBtn');
    const hideJsonEditorBtn = document.getElementById('hideJsonEditorBtn');

    showJsonEditorBtn.addEventListener('click', () => {
        panelNormal.style.display = 'none';
        jsonEditorContainer.style.display = 'block';
        window.history.pushState(null, '', '?view=editor');
    });

    hideJsonEditorBtn.addEventListener('click', () => {
        panelNormal.style.display = 'block';
        jsonEditorContainer.style.display = 'none';
        window.history.pushState(null, '', window.location.pathname);
    });
    
    // Si la URL ya tiene ?view=editor (por ej. después de un error de guardado), muestra el editor
    if (new URLSearchParams(window.location.search).get('view') === 'editor') {
        showJsonEditorBtn.click();
    }

    // Lógica de búsqueda y reemplazo
    const editor = document.getElementById('json_content_editor');
    const findReplaceBar = document.getElementById('findReplaceBar');
    const findInput = document.getElementById('findInput');
    const replaceInput = document.getElementById('replaceInput');
    const replaceBtn = document.getElementById('replaceBtn');
    const replaceAllBtn = document.getElementById('replaceAllBtn');
    const closeFindBtn = document.getElementById('closeFindBtn');

    editor.addEventListener('keydown', (e) => {
        if (e.ctrlKey && e.key === 'f') {
            e.preventDefault();
            findReplaceBar.style.display = 'block';
            findInput.focus();
        }
    });

    closeFindBtn.addEventListener('click', () => {
        findReplaceBar.style.display = 'none';
    });

    replaceBtn.addEventListener('click', () => {
        const findValue = findInput.value;
        const replaceValue = replaceInput.value;
        if (findValue) {
            editor.value = editor.value.replace(findValue, replaceValue);
        }
    });
    
    replaceAllBtn.addEventListener('click', () => {
        const findValue = findInput.value;
        const replaceValue = replaceInput.value;
        if (findValue) {
            // Usamos un RegExp con el flag 'g' para reemplazar todas las ocurrencias
            const regex = new RegExp(findValue.replace(/[-\/\\^$*+?.()|[\]{}]/g, '\\$&'), 'g');
            editor.value = editor.value.replace(regex, replaceValue);
        }
    });
});
</script>
</body>
</html>